/*
 * Copyright (c) 2008-2017 wetator.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package org.wetator.testeditor.editors.xsd;

import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.EditingSupport;
import org.wetator.scripter.XMLScripter;

/**
 * Abstract editing support for {@link WTEXSDTableViewer}.
 *
 * @author tobwoerk
 */
public abstract class AbstractWTEXSDEditingSupport extends EditingSupport {

  private int index;

  /**
   * Constructor.
   *
   * @param aViewer the table viewer
   * @param anIndex the index of the column
   */
  public AbstractWTEXSDEditingSupport(final WTEXSDTableViewer aViewer, final int anIndex) {
    super(aViewer);
    index = anIndex;
  }

  @Override
  protected CellEditor getCellEditor(final Object anElement) {
    return getViewer().getCellEditors()[index];
  }

  @Override
  protected void setValue(final Object anElement, final Object aValue) {
    final WetatorXMLSchema tmpSchema = (WetatorXMLSchema) anElement;
    setValue(tmpSchema, aValue);
    getViewer().refresh();
  }

  /**
   * Convenience setValue possibility for {@link WetatorXMLSchema} elements.
   *
   * @param aSchema the schema
   * @param aValue the value to set
   */
  protected abstract void setValue(WetatorXMLSchema aSchema, Object aValue);

  @Override
  protected Object getValue(final Object anElement) {
    final WetatorXMLSchema tmpSchema = (WetatorXMLSchema) anElement;
    return getValue(tmpSchema);
  }

  /**
   * Convenience getValue possibility for {@link WetatorXMLSchema} elements.
   *
   * @param aSchema the schema to get value from
   * @return the value
   */
  protected abstract Object getValue(WetatorXMLSchema aSchema);

  /**
   * {@inheritDoc}
   *
   * @see org.eclipse.jface.viewers.EditingSupport#canEdit(java.lang.Object)
   */
  @Override
  protected boolean canEdit(final Object anElement) {
    final WetatorXMLSchema tmpSchema = (WetatorXMLSchema) anElement;
    return canEdit(tmpSchema);
  }

  /**
   * Convenience canEdit possibility for {@link WetatorXMLSchema} elements.
   *
   * @param aSchema the schema to check
   * @return true as default, false for base and default schema
   */
  protected boolean canEdit(final WetatorXMLSchema aSchema) {
    if (XMLScripter.BASE_SCHEMA.getNamespace().equals(aSchema.getNamespace())
        || XMLScripter.DEFAULT_COMMAND_SET_SCHEMA.getNamespace().equals(aSchema.getNamespace())) {
      return false;
    }
    return true;
  }
}