/*
 * Copyright (c) 2008-2017 wetator.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package org.wetator.testeditor.editors.xml;

import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.ITextDoubleClickStrategy;
import org.eclipse.jface.text.TextAttribute;
import org.eclipse.jface.text.presentation.IPresentationReconciler;
import org.eclipse.jface.text.presentation.PresentationReconciler;
import org.eclipse.jface.text.rules.DefaultDamagerRepairer;
import org.eclipse.jface.text.rules.Token;
import org.eclipse.jface.text.source.ISourceViewer;
import org.eclipse.jface.text.source.SourceViewerConfiguration;
import org.wetator.testeditor.editors.xml.scanner.XMLCommentScanner;
import org.wetator.testeditor.editors.xml.scanner.XMLPartitionScanner;
import org.wetator.testeditor.editors.xml.scanner.XMLScanner;

/**
 * The configuration for the {@link XMLEditor}.
 * 
 * @author eclipse
 * @author tobwoerk
 */
public class XMLEditorConfiguration extends SourceViewerConfiguration {
  private XMLDoubleClickStrategy doubleClickStrategy;
  private XMLCommentScanner tagScanner;
  private XMLScanner scanner;
  private XMLColorManager colorManager;

  /**
   * Constructor setting the colorManager.
   * 
   * @param aColorManager the colorManager to set
   */
  public XMLEditorConfiguration(final XMLColorManager aColorManager) {
    colorManager = aColorManager;
  }

  /**
   * @see org.eclipse.jface.text.source.SourceViewerConfiguration#getConfiguredContentTypes(org.eclipse.jface.text.source.ISourceViewer)
   * @param aSourceViewer the source viewer
   * @return the configured content types
   */
  @Override
  public String[] getConfiguredContentTypes(final ISourceViewer aSourceViewer) {
    return new String[] { IDocument.DEFAULT_CONTENT_TYPE, XMLPartitionScanner.XML_COMMENT, XMLPartitionScanner.XML_TAG };
  }

  /**
   * @see org.eclipse.jface.text.source.SourceViewerConfiguration#getDoubleClickStrategy(org.eclipse.jface.text.source.ISourceViewer,
   *      java.lang.String)
   * @param aSourceViewer the source viewer
   * @param aContentType the content type
   * @return the double click strategy
   */
  @Override
  public ITextDoubleClickStrategy getDoubleClickStrategy(final ISourceViewer aSourceViewer, final String aContentType) {
    if (doubleClickStrategy == null) {
      doubleClickStrategy = new XMLDoubleClickStrategy();
    }
    return doubleClickStrategy;
  }

  private XMLScanner getXMLScanner() {
    if (scanner == null) {
      scanner = new XMLScanner(colorManager);
      scanner.setDefaultReturnToken(new Token(new TextAttribute(colorManager.getColor(XMLColorConstants.DEFAULT))));
    }
    return scanner;
  }

  private XMLCommentScanner getXMLTagScanner() {
    if (tagScanner == null) {
      tagScanner = new XMLCommentScanner(colorManager);
      tagScanner.setDefaultReturnToken(new Token(new TextAttribute(colorManager.getColor(XMLColorConstants.TAG))));
    }
    return tagScanner;
  }

  /**
   * @see org.eclipse.jface.text.source.SourceViewerConfiguration#getPresentationReconciler(org.eclipse.jface.text.source.ISourceViewer)
   * @param aSourceViewer the source viewer
   * @return the presentation reconciler
   */
  @Override
  public IPresentationReconciler getPresentationReconciler(final ISourceViewer aSourceViewer) {
    final PresentationReconciler tmpReconciler = new PresentationReconciler();

    DefaultDamagerRepairer tmpRepairer = new DefaultDamagerRepairer(getXMLTagScanner());
    tmpReconciler.setDamager(tmpRepairer, XMLPartitionScanner.XML_TAG);
    tmpReconciler.setRepairer(tmpRepairer, XMLPartitionScanner.XML_TAG);

    tmpRepairer = new DefaultDamagerRepairer(getXMLScanner());
    tmpReconciler.setDamager(tmpRepairer, IDocument.DEFAULT_CONTENT_TYPE);
    tmpReconciler.setRepairer(tmpRepairer, IDocument.DEFAULT_CONTENT_TYPE);

    final NonRuleBasedDamagerRepairer tmpNonRuleBasedRepairer = new NonRuleBasedDamagerRepairer(new TextAttribute(
        colorManager.getColor(XMLColorConstants.XML_COMMENT)));
    tmpReconciler.setDamager(tmpNonRuleBasedRepairer, XMLPartitionScanner.XML_COMMENT);
    tmpReconciler.setRepairer(tmpNonRuleBasedRepairer, XMLPartitionScanner.XML_COMMENT);

    return tmpReconciler;
  }
}