/*
 * Copyright (c) 2008-2017 wetator.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package org.wetator.testeditor.editors;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;

import org.apache.commons.lang3.StringUtils;
import org.wetator.core.Command;
import org.wetator.core.IScripter;
import org.wetator.exception.InvalidInputException;
import org.wetator.scripter.LegacyXMLScripter;
import org.wetator.scripter.XMLScripter;
import org.wetator.scripter.xml.ModelBuilder;
import org.wetator.scripter.xml.SchemaFinder;
import org.wetator.scripter.xml.XMLSchema;
import org.wetator.scripter.xml.model.CommandType;
import org.wetator.scripter.xml.model.ParameterType;
import org.wetator.testeditor.editors.exception.CommandNotSupportedException;
import org.wetator.testeditor.editors.exception.FileNotSupportedException;
import org.wetator.testeditor.editors.wte.WTETableViewer;
import org.wetator.testeditor.editors.wte.WetatorCommand;
import org.wetator.testeditor.editors.wte.WetatorCommandList;
import org.wetator.testeditor.editors.xsd.WetatorSchemaList;
import org.wetator.testeditor.editors.xsd.WetatorXMLSchema;
import org.wetator.testeditor.util.WTEUtil;

/**
 * The manager for the test content, managing a commandList for the {@link WTETableViewer}, a schemaList for the
 * {@link org.wetator.testeditor.editors.xsd.WTEXSDTableViewer} and providing XML converting functionality for the whole
 * test-case.
 *
 * @author tobwoerk
 * @author frank.danek
 */
public class WetatorTestContentManager {

  private static final String XML_ENCODING = "UTF-8";
  private static final String TEST_CASE_XSD_VERSION = "1.0.0";

  private static final WetatorXMLSchema BASE_SCHEMA = new WetatorXMLSchema(XMLScripter.BASE_SCHEMA);
  private static final WetatorXMLSchema DEFAULT_COMMAND_SET_SCHEMA = new WetatorXMLSchema(
      XMLScripter.DEFAULT_COMMAND_SET_SCHEMA);
  /** The schema for the SQL command set. */
  public static final WetatorXMLSchema SQL_COMMAND_SET_SCHEMA = new WetatorXMLSchema("sql",
      "http://www.wetator.org/xsd/sql-command-set", "sql-command-set-1.0.0.xsd");
  /** The schema for the test command set. */
  public static final WetatorXMLSchema TEST_COMMAND_SET_SCHEMA = new WetatorXMLSchema("tst",
      "http://www.wetator.org/xsd/test-command-set", "test-command-set-1.0.0.xsd");
  /** The schema for the incubator command set. */
  public static final WetatorXMLSchema INCUBATOR_COMMAND_SET_SCHEMA = new WetatorXMLSchema("inc",
      "http://www.wetator.org/xsd/incubator-command-set", "incubator-command-set-1.0.0.xsd");

  /**
   * The element name for the root element test case.
   */
  public static final String E_TEST_CASE = "test-case";
  /**
   * The element name for test step.
   */
  public static final String E_STEP = "step";
  /**
   * The element name for optional parameter.
   */
  public static final String E_OPTIONAL_PARAMETER = "optionalParameter";
  /**
   * The element name for second optional parameter.
   */
  public static final String E_OPTIONAL_PARAMETER2 = "optionalParameter2";
  /**
   * The element name for commands.
   */
  public static final String E_COMMAND = "command";
  /**
   * The element name for comments.
   */
  public static final String E_COMMENT = "comment";
  private static final String A_DISABLED = "disabled";

  private File file;

  private List<String> availableCommands = new ArrayList<String>();

  private XMLScripter scripter;
  private Map<String, String> namespacePrefixes = new HashMap<String, String>();
  private List<Command> wetatorCommands;
  private List<WetatorCommand> commands;
  private final WetatorSchemaList schemaList = new WetatorSchemaList();
  private final WetatorCommandList commandList = new WetatorCommandList();

  private static final Pattern CHARACTER_DATA_PATTERN = Pattern.compile(".*[<>&]");

  /**
   * Initialize the content from the file with the given path.
   *
   * @param aTestFilePath the path to the file holding the test content (in Wetator XML format)
   * @return true if initialization went fine, false if conversion is necessary
   * @throws InvalidInputException if (dirty) test file cannot be scripted
   * @throws FileNotSupportedException if test file cannot be opened
   */
  public boolean initialize(final String aTestFilePath) throws InvalidInputException, FileNotSupportedException {
    scripter = new XMLScripter();
    return initialize(false, aTestFilePath, null);
  }

  /**
   * Initialize the content.<br/>
   * Content manager must have been initialized with {@link #initialize(String)} before.
   *
   * @param aDirty flag marking if only dirty initialization is possible
   * @param aTestFilePath the path to the file holding the test content (in Wetator XML format)
   * @param aTestContent the current test content
   * @return true if initialization went fine, false if conversion is necessary
   * @throws InvalidInputException if (dirty) test file cannot be scripted
   * @throws FileNotSupportedException if test file cannot be opened
   */
  public boolean initialize(final boolean aDirty, final String aTestFilePath, final String aTestContent)
      throws InvalidInputException, FileNotSupportedException {
    file = new File(aTestFilePath);
    if (aDirty) {
      scripter.script(aTestContent, file.getParentFile());
    } else {
      try {
        // full reload
        if (IScripter.IS_SUPPORTED == scripter.isSupported(file)) {
          scripter.script(file);
        } else {
          final LegacyXMLScripter tmpLegacyScripter = new LegacyXMLScripter();
          if (IScripter.IS_SUPPORTED == tmpLegacyScripter.isSupported(file)) {
            tmpLegacyScripter.script(file);
            wetatorCommands = tmpLegacyScripter.getCommands();
            commands = readCommands();
            loadCommands();
            return false;
          }
          throw new FileNotSupportedException("No scripter found for file '" + aTestFilePath + "'.");
        }
      } catch (final InvalidInputException e) {
        throw new FileNotSupportedException("Could not script file '" + aTestFilePath + "'.");
      }
    }
    reload();
    return true;
  }

  private void reload() {
    for (final XMLSchema tmpSchema : scripter.getSchemas()) {
      namespacePrefixes.put(tmpSchema.getNamespace(), tmpSchema.getPrefix());
    }
    wetatorCommands = scripter.getCommands();
    commands = readCommands();

    loadSchemas();
    availableCommands.clear();
    loadCommands();
  }

  private void loadSchemas() {
    schemaList.initData(WTEUtil.convertToWetatorXMLSchemaList(scripter.getSchemas()));
    schemaList.getElements().add(new WetatorXMLSchema("", "", ""));
  }

  private void loadCommands() {
    commandList.initData(commands);
    commandList.getElements().add(new WetatorCommand("", "", "", "", Boolean.FALSE));
  }

  private List<WetatorCommand> readCommands() {
    final List<WetatorCommand> tmpResult = new LinkedList<WetatorCommand>();

    for (final Command tmpWetCommand : wetatorCommands) {
      String tmpCommandName = "";
      if (null != tmpWetCommand.getName()) {
        tmpCommandName = tmpWetCommand.getName();
      }
      String tmpFirstParam = "";
      if (null != tmpWetCommand.getFirstParameter()) {
        tmpFirstParam = tmpWetCommand.getFirstParameter().getValue();
      }
      String tmpSecondParam = "";
      if (null != tmpWetCommand.getSecondParameter()) {
        tmpSecondParam = tmpWetCommand.getSecondParameter().getValue();
      }
      String tmpThirdParam = "";
      if (null != tmpWetCommand.getThirdParameter()) {
        tmpThirdParam = tmpWetCommand.getThirdParameter().getValue();
      }
      final WetatorCommand tmpCommand = new WetatorCommand(tmpCommandName, tmpFirstParam, tmpSecondParam, tmpThirdParam,
          tmpWetCommand.isComment());
      tmpResult.add(tmpCommand);
    }

    return tmpResult;
  }

  /**
   * The test case with the current modifications as XML.
   *
   * @return the test case as XML
   * @throws CommandNotSupportedException if test case contains an unsupported command
   * @throws XMLStreamException in case of bugs in this method ;)
   * @throws IOException if reading the file encounters problems
   */
  public String getTestCaseAsXml() throws CommandNotSupportedException, XMLStreamException, IOException {
    final StringBuilder tmpXml = new StringBuilder(150);

    tmpXml.append("<?xml version='1.0' encoding='").append(XML_ENCODING).append("'?>\n<").append(E_TEST_CASE);

    final StringBuilder tmpLocations = new StringBuilder();
    ModelBuilder tmpModel = null;
    if (schemaList.getElements().isEmpty()) {
      determineNeededSchemasFromOldTestFile();
      for (final WetatorXMLSchema tmpSchema : schemaList.getElements()) {
        namespacePrefixes.put(tmpSchema.getNamespace(), tmpSchema.getPrefix());
      }
      try {
        tmpModel = new ModelBuilder(WTEUtil.convertToXMLSchemaList(schemaList.getElements()), file.getParentFile());
      } catch (final Exception e) {
        throw new RuntimeException(e);
      }
    } else {
      tmpModel = scripter.getModel();
    }
    final List<XMLSchema> tmpWrittenSchemas = new ArrayList<XMLSchema>();
    for (final WetatorXMLSchema tmpSchema : schemaList.getElements()) {
      if (tmpSchema.hasData() && !tmpWrittenSchemas.contains(tmpSchema.getInnerSchema())) {
        tmpXml.append("\n    xmlns");
        if (null != tmpSchema.getPrefix()) {
          tmpXml.append(':');
          tmpXml.append(tmpSchema.getPrefix());
        }
        tmpXml.append("='");
        tmpXml.append(tmpSchema.getNamespace());
        tmpXml.append('\'');

        tmpLocations.append(tmpSchema.getNamespace());
        tmpLocations.append(' ');
        tmpLocations.append(tmpSchema.getLocation());
        tmpLocations.append(' ');
        tmpWrittenSchemas.add(tmpSchema.getInnerSchema());
      }
    }
    tmpXml.append("\n    xmlns:xsi='http://www.w3.org/2001/XMLSchema-instance'\n    xsi:schemaLocation='");
    tmpXml.append(tmpLocations.substring(0, tmpLocations.length() - 1));
    tmpXml.append("'\n    version='");
    tmpXml.append(TEST_CASE_XSD_VERSION);
    tmpXml.append("'>\n");

    final XMLOutputFactory tmpFactory = XMLOutputFactory.newInstance();
    final ByteArrayOutputStream tmpStream = new ByteArrayOutputStream();
    final XMLStreamWriter tmpWriter = tmpFactory.createXMLStreamWriter(tmpStream, XML_ENCODING);
    if (!commandList.getElements().isEmpty()) {
      for (final WetatorCommand tmpWetatorCommand : commandList.getElements()) {
        if (tmpWetatorCommand.hasData()) {
          final Command tmpCommand = tmpWetatorCommand.getInnerCommand();

          tmpWriter.writeCharacters("    ");
          if (tmpCommand.isComment() && StringUtils.isEmpty(tmpCommand.getName())) {
            writeComment(tmpWriter, tmpCommand);
          } else {
            CommandType tmpCommandType = null;
            if ("".equals(tmpCommand.getName())) {
              writeComment(tmpWriter, tmpCommand);
            } else {
              tmpCommandType = tmpModel.getCommandType(tmpCommand.getName());
              if (tmpCommandType == null) {
                throw new CommandNotSupportedException("Unknown command '" + tmpCommand.getName() + "'.");
              }
              tmpWriter.writeStartElement(E_COMMAND);
              if (tmpCommand.isComment()) {
                tmpWriter.writeAttribute(A_DISABLED, "true");
              }

              final String tmpPrefix = namespacePrefixes.get(tmpCommandType.getNamespace());
              if (tmpPrefix == null) {
                throw new RuntimeException("Unknown namespace '" + tmpCommandType.getNamespace() + "'.");
              }
              tmpWriter.writeStartElement(tmpPrefix, tmpCommandType.getName(), tmpCommandType.getNamespace());

              final Collection<ParameterType> tmpParameterTypes = tmpCommandType.getParameterTypes();
              final String[] tmpParameterValues = new String[tmpParameterTypes.size()];
              if (tmpParameterValues.length >= 1 && tmpCommand.getFirstParameter() != null) {
                tmpParameterValues[0] = tmpCommand.getFirstParameter().getValue();
              }
              if (tmpParameterValues.length >= 2 && tmpCommand.getSecondParameter() != null) {
                tmpParameterValues[1] = tmpCommand.getSecondParameter().getValue();
              }
              if (tmpParameterValues.length >= 3 && tmpCommand.getThirdParameter() != null) {
                tmpParameterValues[2] = tmpCommand.getThirdParameter().getValue();
              }
              int i = 0;
              for (final ParameterType tmpParameterType : tmpParameterTypes) {
                if (StringUtils.isNotEmpty(tmpParameterValues[i])) {
                  tmpWriter.writeStartElement(tmpParameterType.getNamespace(), tmpParameterType.getName());
                  writeContent(tmpWriter, tmpParameterValues[i]);
                  tmpWriter.writeEndElement();
                }
                i++;
              }

              tmpWriter.writeEndElement();
              tmpWriter.writeEndElement();
              tmpWriter.writeCharacters("\n");
            }
          }
        }
      }
      tmpWriter.close();
    }

    tmpXml.append(tmpStream.toString(XML_ENCODING));
    tmpXml.append("</" + E_TEST_CASE + ">");

    return tmpXml.toString();
  }

  private void determineNeededSchemasFromOldTestFile() throws XMLStreamException, IOException {
    schemaList.getElements().add(BASE_SCHEMA);
    schemaList.getElements().add(DEFAULT_COMMAND_SET_SCHEMA);
    final Reader tmpReader = new InputStreamReader(new FileInputStream(file), XML_ENCODING);
    final SchemaFinder tmpFinder = new SchemaFinder(tmpReader);
    final List<XMLSchema> tmpOldSchemas = tmpFinder.getSchemas();
    for (final XMLSchema tmpOldSchema : tmpOldSchemas) {
      if ("http://www.wetator.org/xsd/sqlCommandSet".equals(tmpOldSchema.getNamespace())) {
        schemaList.getElements().add(SQL_COMMAND_SET_SCHEMA);
      } else if ("http://www.wetator.org/xsd/testCommandSet".equals(tmpOldSchema.getNamespace())) {
        schemaList.getElements().add(TEST_COMMAND_SET_SCHEMA);
      } else if ("http://www.wetator.org/xsd/incubatorCommandSet".equals(tmpOldSchema.getNamespace())) {
        schemaList.getElements().add(INCUBATOR_COMMAND_SET_SCHEMA);
      }
    }
  }

  private void writeComment(final XMLStreamWriter aWriter, final Command aCommand) throws XMLStreamException {
    aWriter.writeStartElement(E_COMMENT);
    if (aCommand.getFirstParameter() != null) {
      writeContent(aWriter, aCommand.getFirstParameter().getValue());
    }
    aWriter.writeEndElement();
    aWriter.writeCharacters("\n");
  }

  private void writeContent(final XMLStreamWriter aWriter, final String aContent) throws XMLStreamException {
    if (CHARACTER_DATA_PATTERN.matcher(aContent).matches()) {
      aWriter.writeCData(aContent);
    } else {
      aWriter.writeCharacters(aContent);
    }
  }

  /**
   * @param aCommand the command to get documentation for
   * @param anIndex the index of the command element documentation that should be returned
   * @return the documentation for the given command at the given index
   */
  public TagInformation getTagInformation(final Command aCommand, final int anIndex) {
    final CommandType tmpCommandType = scripter.getModel().getCommandType(aCommand.getName());

    if (null == tmpCommandType || WTETableViewer.IS_COMMENT == anIndex) {
      return null;
    }

    final TagInformation tmpTagInfo = new TagInformation();
    if (WTETableViewer.COMMAND == anIndex) {
      tmpTagInfo.setName(tmpCommandType.getName());
      tmpTagInfo.setDocumentation(tmpCommandType.getDocumentation());
      return tmpTagInfo;
    }

    final List<ParameterType> tmpParameterTypes = tmpCommandType.getParameterTypes();
    final int tmpParameterIndex;
    ParameterType tmpParameterType = null;
    if (WTETableViewer.PARAMETER == anIndex) {
      tmpParameterIndex = 0;
    } else if (WTETableViewer.OPTIONAL_PARAMETER == anIndex) {
      tmpParameterIndex = 1;
    } else if (WTETableViewer.OPTIONAL_PARAMETER2 == anIndex) {
      tmpParameterIndex = 2;
    } else {
      throw new IllegalArgumentException("Invalid index for getting tag information: " + anIndex);
    }

    if (tmpParameterTypes.size() > tmpParameterIndex) {
      tmpParameterType = tmpParameterTypes.get(tmpParameterIndex);
    } else {
      return null;
    }

    if (null == tmpParameterType) {
      throw new IllegalArgumentException(
          "No parameter found for command '" + tmpCommandType.getName() + "'at index: " + tmpParameterIndex);
    }

    tmpTagInfo.setName(tmpParameterType.getName());
    tmpTagInfo.setDocumentation(tmpParameterType.getDocumentation());
    return tmpTagInfo;
  }

  /**
   * Returns the available commands for editing Wetator test files.<br/>
   * Caches and therefore loads the commands if necessary.<br/>
   * Adds an empty command on top for displaying it in a dropdown list.
   *
   * @return the available commands
   */
  public String[] getAvailableCommands() {
    if (availableCommands.size() == 0) {
      availableCommands.add("");
      for (final CommandType tmpCommandType : scripter.getModel().getCommandTypes()) {
        availableCommands.add(tmpCommandType.getName());
      }
      Collections.sort(availableCommands);
    }
    return availableCommands.toArray(new String[availableCommands.size()]);
  }

  /**
   * @return true if at least one command type needs 3 parameters
   */
  public boolean isAnyCommandTypeWith3Parameters() {
    for (final CommandType tmpCommandType : scripter.getModel().getCommandTypes()) {
      if (tmpCommandType.getParameterTypes().size() == 3) {
        return true;
      }
    }
    return false;
  }

  /**
   * @param aCommandName the name of the command to check
   * @return true if the given command type needs 3 parameters
   */
  public boolean isCommandTypeWith3Parameters(final String aCommandName) {
    for (final CommandType tmpCommandType : scripter.getModel().getCommandTypes()) {
      if (tmpCommandType.getParameterTypes().size() == 3 && tmpCommandType.getName().equals(aCommandName)) {
        return true;
      }
    }
    return false;
  }

  /**
   * @return the schemaList
   */
  public WetatorSchemaList getSchemaList() {
    return schemaList;
  }

  /**
   * @return the commandList
   */
  public WetatorCommandList getCommandList() {
    return commandList;
  }
}