/*
 * Copyright (c) 2008-2017 wetator.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package org.wetator.testeditor.editors;

import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.Vector;

/**
 * Abstract content list presented in WTE table viewers.
 * 
 * @param <T> the type of objects that are the content
 * @author tobwoerk
 * @author frank.danek
 */
public abstract class AbstractWTEContentList<T extends AbstractWTETableViewerElement> {

  private static final int COUNT = 10;
  /** The element {@link List} holding the content. */
  protected List<T> elements = new Vector<T>(COUNT);
  private Set<AbstractWTETableViewer<T>.WTETableViewerContentProvider> changeListeners = new HashSet<AbstractWTETableViewer<T>.WTETableViewerContentProvider>();

  private final Class<T> type;

  /**
   * Constructor.
   * 
   * @param aType the type of objects that are the content
   */
  protected AbstractWTEContentList(final Class<T> aType) {
    super();
    type = aType;
  }

  /**
   * @param anElementList the elements to initialize with
   */
  public void initData(final List<T> anElementList) {
    elements.clear();
    for (final T tmpElement : anElementList) {
      elements.add(tmpElement);
    }
  }

  /**
   * @return the elements
   */
  public List<T> getElements() {
    return elements;
  }

  /**
   * Add a new element to the collection of elements at the position of the given element.
   * 
   * @param anElement the element defining the index for the new one
   */
  public void addNewElementBefore(final T anElement) {
    final T tmpElement = getNewElement();
    final int tmpIndex = elements.indexOf(anElement);
    elements.add(tmpIndex, tmpElement);
  }

  /**
   * Add a new element to the collection of elements after the position of the given element.
   * 
   * @param anElement the element defining where to put the new one
   */
  private void addNewElementAfter(final T anElement) {
    final T tmpElement = getNewElement();
    final int tmpIndex = elements.indexOf(anElement) + 1;
    elements.add(tmpIndex, tmpElement);
  }

  private T getNewElement() {
    try {
      return type.newInstance();
    } catch (final Exception e) {
      throw new RuntimeException(e);
    }
  }

  /**
   * @param anElement the element to remove
   */
  public void removeElement(final T anElement) {
    elements.remove(anElement);
  }

  /**
   * @param anElement the element to change
   */
  public void elementChanged(final T anElement) {
    final Iterator<AbstractWTETableViewer<T>.WTETableViewerContentProvider> tmpIterator = changeListeners.iterator();
    while (tmpIterator.hasNext()) {
      tmpIterator.next().updateElement(anElement);
    }

    addEmptyEndElementIfNecessary(anElement);
  }

  /**
   * Adds an empty line at the end of the test file if necessary.
   * 
   * @param anElement the element to check (may be the last one of the element list)
   */
  public void addEmptyEndElementIfNecessary(final T anElement) {
    final int tmpIndex = elements.indexOf(anElement);
    // if last element got data, add new one at the end of the list
    if (tmpIndex == elements.size() - 1 && anElement.hasData()) {
      addNewElementAfter(anElement);
    }
  }

  /**
   * @param aViewer the viewer to remove from the change listeners
   */
  public void removeChangeListener(final AbstractWTETableViewer<T>.WTETableViewerContentProvider aViewer) {
    changeListeners.remove(aViewer);
  }

  /**
   * @param aViewer the viewer to add to the change listeners
   */
  public void addChangeListener(final AbstractWTETableViewer<T>.WTETableViewerContentProvider aViewer) {
    changeListeners.add(aViewer);
  }
}