/*
 * Copyright (c) 2008-2017 wetator.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package org.wetator.testeditor.editors.xml;

import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.ITextDoubleClickStrategy;
import org.eclipse.jface.text.ITextViewer;

/**
 * The double click strategy for the {@link XMLEditorConfiguration}.
 *
 * @author eclipse
 * @author tobwoerk
 */
public class XMLDoubleClickStrategy implements ITextDoubleClickStrategy {

  private ITextViewer fText;

  /**
   * @see org.eclipse.jface.text.ITextDoubleClickStrategy#doubleClicked(org.eclipse.jface.text.ITextViewer)
   * @param aPart the part that was double clicked
   */
  @Override
  public void doubleClicked(final ITextViewer aPart) {
    final int tmpPos = aPart.getSelectedRange().x;

    if (tmpPos < 0) {
      return;
    }

    fText = aPart;

    if (!selectComment(tmpPos)) {
      selectWord(tmpPos);
    }
  }

  /**
   * @param aCaretPos the position
   * @return true if comment selected, false if not
   */
  protected boolean selectComment(final int aCaretPos) {
    final IDocument tmpDocument = fText.getDocument();
    final int tmpStartPos;
    final int tmpEndPos;

    try {
      int tmpPosition = aCaretPos;
      char tmpCharacter = ' ';

      while (tmpPosition >= 0) {
        tmpCharacter = tmpDocument.getChar(tmpPosition);
        if (tmpCharacter == '\\') {
          tmpPosition -= 2;
          continue;
        }
        if (tmpCharacter == Character.LINE_SEPARATOR || tmpCharacter == '\"') {
          break;
        }
        --tmpPosition;
      }

      if (tmpCharacter != '\"') {
        return false;
      }

      tmpStartPos = tmpPosition;

      tmpPosition = aCaretPos;
      final int tmpLength = tmpDocument.getLength();
      tmpCharacter = ' ';

      while (tmpPosition < tmpLength) {
        tmpCharacter = tmpDocument.getChar(tmpPosition);
        if (tmpCharacter == Character.LINE_SEPARATOR || tmpCharacter == '\"') {
          break;
        }
        ++tmpPosition;
      }
      if (tmpCharacter != '\"') {
        return false;
      }

      tmpEndPos = tmpPosition;

      final int tmpOffset = tmpStartPos + 1;
      final int tmpLen = tmpEndPos - tmpOffset;
      fText.setSelectedRange(tmpOffset, tmpLen);
      return true;
    } catch (final BadLocationException e) {
      // bad location, bad luck
    }

    return false;
  }

  /**
   * @param aCaretPos the position
   * @return true if word selected, false if not
   */
  protected boolean selectWord(final int aCaretPos) {
    final IDocument tmpDocument = fText.getDocument();
    final int tmpStartPos;
    final int tmpEndPos;

    try {

      int tmpPosition = aCaretPos;
      char tmpCharacter;

      while (tmpPosition >= 0) {
        tmpCharacter = tmpDocument.getChar(tmpPosition);
        if (!Character.isJavaIdentifierPart(tmpCharacter)) {
          break;
        }
        --tmpPosition;
      }

      tmpStartPos = tmpPosition;

      tmpPosition = aCaretPos;
      final int tmpLength = tmpDocument.getLength();

      while (tmpPosition < tmpLength) {
        tmpCharacter = tmpDocument.getChar(tmpPosition);
        if (!Character.isJavaIdentifierPart(tmpCharacter)) {
          break;
        }
        ++tmpPosition;
      }

      tmpEndPos = tmpPosition;
      selectRange(tmpStartPos, tmpEndPos);
      return true;

    } catch (final BadLocationException e) {
      // bad location, bad luck
    }

    return false;
  }

  private void selectRange(final int aStartPos, final int aStopPos) {
    final int tmpOffset = aStartPos + 1;
    final int tmpLength = aStopPos - tmpOffset;
    fText.setSelectedRange(tmpOffset, tmpLength);
  }
}