/*
 * Copyright (c) 2008-2017 wetator.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package org.wetator.testeditor.editors.xml;

import org.eclipse.core.runtime.Assert;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.DocumentEvent;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.IRegion;
import org.eclipse.jface.text.ITypedRegion;
import org.eclipse.jface.text.Region;
import org.eclipse.jface.text.TextAttribute;
import org.eclipse.jface.text.TextPresentation;
import org.eclipse.jface.text.presentation.IPresentationDamager;
import org.eclipse.jface.text.presentation.IPresentationRepairer;
import org.eclipse.swt.custom.StyleRange;

/**
 * Repairs damaged areas in the presentation.
 *
 * @author eclipse
 * @author tobwoerk
 */
public class NonRuleBasedDamagerRepairer implements IPresentationDamager, IPresentationRepairer {

  /** The document this object works on. */
  protected IDocument fDocument;
  /** The default text attribute if non is returned as data by the current token. */
  protected TextAttribute fDefaultTextAttribute;

  /**
   * Constructor for NonRuleBasedDamagerRepairer.
   *
   * @param aDefaultTextAttribute the default text attribute
   */
  public NonRuleBasedDamagerRepairer(final TextAttribute aDefaultTextAttribute) {
    Assert.isNotNull(aDefaultTextAttribute);

    fDefaultTextAttribute = aDefaultTextAttribute;
  }

  /**
   * @see IPresentationRepairer#setDocument(IDocument)
   * @param aDocument the document to set
   */
  @Override
  public void setDocument(final IDocument aDocument) {
    fDocument = aDocument;
  }

  /**
   * Returns the end offset of the line that contains the specified offset or
   * if the offset is inside a line delimiter, the end offset of the next line.
   *
   * @param anOffset the offset whose line end offset must be computed
   * @return the line end offset for the given offset
   * @throws BadLocationException if offset is invalid in the current document
   */
  protected int endOfLineOf(final int anOffset) throws BadLocationException {
    IRegion tmpInfo = fDocument.getLineInformationOfOffset(anOffset);
    if (anOffset <= tmpInfo.getOffset() + tmpInfo.getLength()) {
      return tmpInfo.getOffset() + tmpInfo.getLength();
    }

    final int tmpLine = fDocument.getLineOfOffset(anOffset);
    try {
      tmpInfo = fDocument.getLineInformation(tmpLine + 1);
      return tmpInfo.getOffset() + tmpInfo.getLength();
    } catch (final BadLocationException e) {
      return fDocument.getLength();
    }
  }

  /**
   * @see IPresentationDamager#getDamageRegion(ITypedRegion, DocumentEvent, boolean)
   * @param aPartition the partition
   * @param anEvent the event
   * @param aDocumentPartitioningChanged flag marking change
   * @return the damaged region
   */
  @Override
  public IRegion getDamageRegion(final ITypedRegion aPartition, final DocumentEvent anEvent,
      final boolean aDocumentPartitioningChanged) {
    if (!aDocumentPartitioningChanged) {
      try {

        final IRegion tmpInfo = fDocument.getLineInformationOfOffset(anEvent.getOffset());
        final int tmpStart = Math.max(aPartition.getOffset(), tmpInfo.getOffset());

        int tmpEnd = anEvent.getOffset();
        if (null == anEvent.getText()) {
          tmpEnd += anEvent.getLength();
        } else {
          tmpEnd += anEvent.getText().length();
        }

        if (tmpInfo.getOffset() <= tmpEnd && tmpEnd <= tmpInfo.getOffset() + tmpInfo.getLength()) {
          // optimize the case of the same line
          tmpEnd = tmpInfo.getOffset() + tmpInfo.getLength();
        } else {
          tmpEnd = endOfLineOf(tmpEnd);
        }

        tmpEnd = Math.min(aPartition.getOffset() + aPartition.getLength(), tmpEnd);
        return new Region(tmpStart, tmpEnd - tmpStart);

      } catch (final BadLocationException e) {
        // bad location, bad luck
      }
    }

    return aPartition;
  }

  /**
   * @see IPresentationRepairer#createPresentation(TextPresentation, ITypedRegion)
   * @param aPresentation the presentation
   * @param aRegion the region
   */
  @Override
  public void createPresentation(final TextPresentation aPresentation, final ITypedRegion aRegion) {
    addRange(aPresentation, aRegion.getOffset(), aRegion.getLength(), fDefaultTextAttribute);
  }

  /**
   * Adds style information to the given text presentation.
   *
   * @param aPresentation the text presentation to be extended
   * @param anOffset the offset of the range to be styled
   * @param aLength the length of the range to be styled
   * @param anAttribute the attribute describing the style of the range to be styled
   */
  protected void addRange(final TextPresentation aPresentation, final int anOffset, final int aLength,
      final TextAttribute anAttribute) {
    if (anAttribute != null) {
      aPresentation.addStyleRange(new StyleRange(anOffset, aLength, anAttribute.getForeground(),
          anAttribute.getBackground(), anAttribute.getStyle()));
    }
  }
}