/*
 * Copyright (c) 2008-2017 wetator.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package org.wetator.testeditor.editors;

import java.util.List;

import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.ColumnViewer;
import org.eclipse.jface.viewers.ColumnViewerEditor;
import org.eclipse.jface.viewers.ColumnViewerEditorActivationEvent;
import org.eclipse.jface.viewers.ColumnViewerEditorActivationListener;
import org.eclipse.jface.viewers.ColumnViewerEditorActivationStrategy;
import org.eclipse.jface.viewers.ColumnViewerEditorDeactivationEvent;
import org.eclipse.jface.viewers.ComboBoxCellEditor;
import org.eclipse.jface.viewers.FocusCellHighlighter;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.TableViewerEditor;
import org.eclipse.jface.viewers.TableViewerFocusCellManager;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerCell;
import org.eclipse.swt.SWT;
import org.eclipse.swt.dnd.Clipboard;
import org.eclipse.swt.dnd.TextTransfer;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Text;
import org.wetator.testeditor.editors.wte.WetatorCommand;
import org.wetator.testeditor.util.WTEUtil;

/**
 * The abstract table viewer for WTE table viewers.
 *
 * @param <T> the type of objects to show in the table viewer
 * @author tobwoerk
 * @author frank.danek
 */
@SuppressWarnings("unchecked")
public abstract class AbstractWTETableViewer<T extends AbstractWTETableViewerElement> extends TableViewer {

  private WetatorTestMultiPageEditor wte;

  private boolean dirty;
  /** Manages the content. */
  protected WetatorTestContentManager contentManager;

  /** Manages focusing. */
  protected final TableViewerFocusCellManager focusCellManager;

  private ControlAdapter resizeAdapter = new ControlAdapter() {
    @Override
    public void controlResized(final ControlEvent anEvent) {
      resizeDynamicWidthColumns();
    }
  };

  /**
   * Constructor.
   *
   * @param aParent the parent
   * @param aContentManager the contentManager holding the test content
   */
  public AbstractWTETableViewer(final Composite aParent, final WetatorTestContentManager aContentManager) {
    // we need to select just a single row, but if one cell of a row is selected the whole row should be marked as
    // selected (this hint is needed for Windows...)
    super(aParent, SWT.SINGLE | SWT.FULL_SELECTION);
    contentManager = aContentManager;

    setUseHashlookup(true);
    setContentProvider(new WTETableViewerContentProvider());
    reloadInput();

    // Set numColumns to 2 for the buttons
    final GridLayout tmpLayout = new GridLayout(2, false);
    tmpLayout.marginWidth = getColumnNames().size();
    aParent.setLayout(tmpLayout);

    final GridData tmpTableGridData = new GridData(GridData.FILL_BOTH);
    tmpTableGridData.grabExcessVerticalSpace = true;
    tmpTableGridData.horizontalSpan = getColumnNames().size();
    getTable().setLayoutData(tmpTableGridData);

    getTable().setLinesVisible(true);
    getTable().setHeaderVisible(true);
    getTable().addControlListener(resizeAdapter);

    createColumns();
    createEditors();

    createButtons();

    initialize();

    // we have to correct the width of the background rectangle (at least for Windows...)
    getControl().addListener(SWT.EraseItem, new Listener() {
      @Override
      public void handleEvent(final Event anEvent) {
        if ((anEvent.detail & SWT.BACKGROUND) > 0) {
          // the background of a cell is painted
          final Rectangle tmpBounds = ((TableItem) anEvent.item).getBounds(anEvent.index);
          final Rectangle tmpRectangle = new Rectangle(tmpBounds.x, tmpBounds.y, tmpBounds.width, tmpBounds.height);

          // for the first column the x coordinate is somehow wrong
          // so we correct it and adjust the width
          if (anEvent.index == 0) {
            tmpRectangle.width += tmpRectangle.x;
            tmpRectangle.x = 0;
          }

          anEvent.gc.fillRectangle(tmpRectangle);
          anEvent.detail &= ~SWT.BACKGROUND;
        }
      }
    });

    focusCellManager = new TableViewerFocusCellManager(this, new EditableFocusCellHighlighter(this));
    final ColumnViewerEditorActivationStrategy tmpActivationSupport = new ColumnViewerEditorActivationStrategy(this) {
      @Override
      protected boolean isEditorActivationEvent(final ColumnViewerEditorActivationEvent anEvent) {
        final boolean tmpIsActivationEvent = isActivatingEditor(anEvent);
        // TODO maybe there's a nicer way to avoid "selection jumping" based on typed letters in the table?!
        // why is this happening anyway??
        if (tmpIsActivationEvent || WTEUtil.isPrintable(anEvent.character)) {
          deactivateKeyEvent(anEvent);
        }
        return tmpIsActivationEvent;
      }
    };

    TableViewerEditor.create(this, focusCellManager, tmpActivationSupport,
        ColumnViewerEditor.TABBING_HORIZONTAL | ColumnViewerEditor.TABBING_MOVE_TO_ROW_NEIGHBOR
            | ColumnViewerEditor.TABBING_VERTICAL | ColumnViewerEditor.KEYBOARD_ACTIVATION);

    final ColumnViewerEditorActivationListener tmpEditorActivationListener = new ColumnViewerEditorActivationListener() {
      @Override
      public void beforeEditorDeactivated(final ColumnViewerEditorDeactivationEvent anEvent) {
        // nothing to do
      }

      @Override
      public void beforeEditorActivated(final ColumnViewerEditorActivationEvent anEvent) {
        // nothing to do
      }

      @Override
      public void afterEditorDeactivated(final ColumnViewerEditorDeactivationEvent anEvent) {
        // nothing to do
      }

      @Override
      public void afterEditorActivated(final ColumnViewerEditorActivationEvent anEvent) {
        final ViewerCell tmpCell = (ViewerCell) anEvent.getSource();
        final int tmpIndex = tmpCell.getColumnIndex();

        // TODO try selection for command editor in WTE if character allows it

        if (getCellEditors()[tmpIndex] instanceof TextCellEditor) {
          final TextCellEditor tmpEditor = (TextCellEditor) getCellEditors()[tmpIndex];
          final Text tmpText = (Text) tmpEditor.getControl();
          if (anEvent.eventType == ColumnViewerEditorActivationEvent.KEY_PRESSED
              && WTEUtil.isPrintable(anEvent.character)) {
            tmpText.setText(Character.toString(anEvent.character));
            tmpText.setSelection(tmpText.getText().length());
          }
        }
      }
    };

    getColumnViewerEditor().addEditorActivationListener(tmpEditorActivationListener);
  }

  /**
   * Hook for initializing additional stuff.
   */
  protected void initialize() {
    // nothing by default
  }

  /**
   * @param anEvent the event to check
   * @return true if the given event should activate the editor (handles text editor activation as default)
   */
  protected boolean isActivatingEditor(final ColumnViewerEditorActivationEvent anEvent) {
    return anEvent.eventType == ColumnViewerEditorActivationEvent.TRAVERSAL
        || anEvent.eventType == ColumnViewerEditorActivationEvent.MOUSE_DOUBLE_CLICK_SELECTION
        || anEvent.eventType == ColumnViewerEditorActivationEvent.KEY_PRESSED
            && (anEvent.keyCode == SWT.CR || anEvent.keyCode == SWT.F2 || WTEUtil.isPrintable(anEvent.character))
        || anEvent.eventType == ColumnViewerEditorActivationEvent.PROGRAMMATIC;
  }

  private void deactivateKeyEvent(final ColumnViewerEditorActivationEvent anEvent) {
    if (anEvent.sourceEvent instanceof KeyEvent) {
      ((KeyEvent) anEvent.sourceEvent).doit = false;
    }
  }

  /**
   * Create the columns of the table viewer.
   */
  protected abstract void createColumns();

  /**
   * The cell highlighter for the WTE table viewers.
   *
   * @author tobwoerk
   * @author frank.danek
   */
  private class EditableFocusCellHighlighter extends FocusCellHighlighter {

    /**
     * @param aViewer
     *        the viewer
     */
    EditableFocusCellHighlighter(final ColumnViewer aViewer) {
      super(aViewer);

      final Listener tmpListener = new Listener() {
        @Override
        public void handleEvent(final Event anEvent) {
          if ((anEvent.detail & SWT.SELECTED) > 0) {
            // a cell of the row was selected so we mark the focused cell
            markCell(anEvent, getFocusCell());
            anEvent.detail &= ~SWT.SELECTED;
          }
          if ((anEvent.detail & SWT.HOT) > 0) {
            // a cell of the row was hovered so we do not mark the focused cell
            markCell(anEvent, null);
            anEvent.detail &= ~SWT.HOT;
          }
        }
      };
      aViewer.getControl().addListener(SWT.EraseItem, tmpListener);
    }

    private void markCell(final Event anEvent, final ViewerCell aFocusCell) {
      final GC tmpGC = anEvent.gc;

      final int tmpCurrentCol = anEvent.index;
      final Rectangle tmpBounds = ((TableItem) anEvent.item).getBounds(tmpCurrentCol);
      final Rectangle tmpRectangle = new Rectangle(tmpBounds.x, tmpBounds.y, tmpBounds.width, tmpBounds.height);

      // for the first column the x coordinate is somehow wrong
      // so we correct it and adjust the with
      if (tmpCurrentCol == 0) {
        tmpRectangle.width += tmpRectangle.x;
        tmpRectangle.x = 0;
      }

      tmpGC.setAlpha(100);
      if (aFocusCell != null && tmpCurrentCol == aFocusCell.getColumnIndex()) {
        // the current cell is the focused cell so we only draw the lines of a rectangle
        tmpGC.setForeground(new Color(Display.getCurrent(), WTEColors.ECLIPSE_BLUE));
        tmpGC.setLineWidth(1);

        // the rectangle is too big so we adjust it
        tmpRectangle.width -= 1;
        tmpRectangle.height -= 1;

        tmpGC.drawRectangle(tmpRectangle);

        // reset for text display
        tmpGC.setForeground(new Color(Display.getCurrent(), WTEColors.BLACK));
      } else {
        // the current cell is not the focused cell so we draw a filled rectangle
        final Color tmpOldBackground = tmpGC.getBackground();
        tmpGC.setBackground(new Color(Display.getCurrent(), WTEColors.ECLIPSE_BLUE));

        tmpGC.fillRectangle(tmpRectangle);

        // reset the background
        tmpGC.setBackground(tmpOldBackground);
      }

      anEvent.detail &= ~SWT.FOCUSED;
    }

    @Override
    protected void focusCellChanged(final ViewerCell aNewCell, final ViewerCell anOldCell) {
      super.focusCellChanged(aNewCell, anOldCell);

      redraw(aNewCell);
      redraw(anOldCell);
    }

    private void redraw(final ViewerCell aCell) {
      if (aCell != null) {
        final Rectangle tmpBounds = aCell.getBounds();
        final Rectangle tmpRectangle = new Rectangle(tmpBounds.x, tmpBounds.y, tmpBounds.width, tmpBounds.height);
        // for the first column the x coordinate is somehow wrong
        // so we correct it and adjust the with
        if (aCell.getColumnIndex() == 0) {
          tmpRectangle.width += tmpRectangle.x;
          tmpRectangle.x = 0;
        }
        aCell.getControl().redraw(tmpRectangle.x, tmpRectangle.y, tmpRectangle.width, tmpRectangle.height, true);
      }
    }
  }

  /**
   * Generic helper for a column.
   *
   * @param anIndex the index of the column to create
   * @param anAlignment the alignment of the column to create
   * @return the created column
   */
  protected TableViewerColumn createColumn(final int anIndex, final int anAlignment) {
    final TableViewerColumn tmpColumn;
    tmpColumn = new TableViewerColumn(this, anAlignment);
    tmpColumn.setLabelProvider(getCellLabelProvider(anIndex));
    tmpColumn.getColumn().setResizable(false);
    tmpColumn.getColumn().setText(getColumnNames().get(anIndex));
    return tmpColumn;
  }

  private void createEditors() {
    final CellEditor[] tmpCellEditors = new CellEditor[getColumnNames().size()];

    initializeEditors(tmpCellEditors);

    setCellEditors(tmpCellEditors);
  }

  /**
   * Implementors initialize their editors and put them into the given array.
   *
   * @param anEditorArray the arrays to put the initialized editors in
   */
  protected abstract void initializeEditors(CellEditor[] anEditorArray);

  private void createButtons() {
    // create and configure the "Add" button
    final Button tmpAdd = new Button(getTableParent(), SWT.PUSH | SWT.CENTER);
    tmpAdd.setText(WTEMessages.buttonAddLine);
    tmpAdd.setToolTipText(WTEMessages.buttonAddLineToolTip);

    GridData tmpGridData = new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING);
    tmpGridData.widthHint = 80;
    tmpAdd.setLayoutData(tmpGridData);

    tmpAdd.addSelectionListener(new SelectionAdapter() {
      @Override
      public void widgetSelected(final SelectionEvent anEvent) {
        addLine();
      }
    });

    // create and configure the "Delete" button
    final Button tmpDelete = new Button(getTableParent(), SWT.PUSH | SWT.CENTER);
    tmpDelete.setText(WTEMessages.buttonDeleteLine);
    tmpDelete.setToolTipText(WTEMessages.buttonDeleteLineToolTip);

    tmpGridData = new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING);
    tmpGridData.widthHint = 80;
    tmpDelete.setLayoutData(tmpGridData);

    tmpDelete.addSelectionListener(new SelectionAdapter() {
      @Override
      public void widgetSelected(final SelectionEvent anEvent) {
        deleteLine();
      }
    });
  }

  /**
   * Copies the text from the selected cell.
   *
   * @param aClipboard the clipboard to write to
   * @return true if something was copied
   */
  public boolean copyFromCell(final Clipboard aClipboard) {
    final ViewerCell tmpFocusCell = focusCellManager.getFocusCell();
    if (null != tmpFocusCell) {
      final int tmpColumnIndex = tmpFocusCell.getColumnIndex();
      final CellEditor tmpEditor = getCellEditors()[tmpColumnIndex];

      final TextTransfer tmpTextTransfer = TextTransfer.getInstance();
      if (tmpEditor instanceof ComboBoxCellEditor) {
        final String tmpCommandName = ((WetatorCommand) tmpFocusCell.getElement()).getCommandName();
        if (!"".equals(tmpCommandName)) {
          aClipboard.setContents(new Object[] { tmpCommandName }, new Transfer[] { tmpTextTransfer });
          return true;
        }
      } else if (tmpEditor instanceof TextCellEditor) {
        final Object tmpData = tmpFocusCell.getText();
        if (!"".equals(tmpData)) {
          if (isCellEditorActive()) {
            tmpEditor.performCopy();
          } else {
            aClipboard.setContents(new Object[] { tmpData }, new Transfer[] { tmpTextTransfer });
          }
          return true;
        }
      }
    }
    return false;
  }

  /**
   * Cuts the text from the selected cell.
   *
   * @param aClipboard the clipboard to write to
   */
  public void cutFromCell(final Clipboard aClipboard) {
    if (copyFromCell(aClipboard)) {
      final ViewerCell tmpFocusCell = focusCellManager.getFocusCell();
      if (null != tmpFocusCell) {
        final int tmpColumnIndex = tmpFocusCell.getColumnIndex();
        final CellEditor tmpEditor = getCellEditors()[tmpColumnIndex];

        if (tmpEditor instanceof TextCellEditor || tmpEditor instanceof ComboBoxCellEditor) {
          final T tmpElement = (T) tmpFocusCell.getViewerRow().getElement();
          if (isCellEditorActive()) {
            tmpEditor.performCut();
          } else {
            setValue(tmpElement, tmpColumnIndex, "");
            refresh();
            setDirty(true);
          }
        }
      }
    }
  }

  /**
   * Deletes the text from the selected cell.
   */
  public void deleteFromCell() {
    final ViewerCell tmpFocusCell = focusCellManager.getFocusCell();
    if (null != tmpFocusCell) {
      final int tmpColumnIndex = tmpFocusCell.getColumnIndex();
      final CellEditor tmpEditor = getCellEditors()[tmpColumnIndex];

      if (tmpEditor instanceof TextCellEditor || tmpEditor instanceof ComboBoxCellEditor) {
        final T tmpElement = (T) tmpFocusCell.getViewerRow().getElement();
        if (isCellEditorActive()) {
          tmpEditor.performDelete();
        } else {
          setValue(tmpElement, tmpColumnIndex, "");
          refresh();
          setDirty(true);
        }
      }
    }
  }

  /**
   * Pastes the currently copied/cut text into the selected cell.
   *
   * @param aClipboard the {@link Clipboard} to get the value from
   */
  public void pasteIntoCell(final Clipboard aClipboard) {
    final ViewerCell tmpFocusCell = focusCellManager.getFocusCell();
    if (null != tmpFocusCell) {
      final int tmpColumnIndex = tmpFocusCell.getColumnIndex();

      // do not paste in the blocked 3 parameters cell
      if (4 == tmpColumnIndex) {
        final String tmpCommandName = ((WetatorCommand) tmpFocusCell.getElement()).getCommandName();
        if (!contentManager.isCommandTypeWith3Parameters(tmpCommandName)) {
          return;
        }
      }

      final CellEditor tmpEditor = getCellEditors()[tmpColumnIndex];

      final TextTransfer tmpTransfer = TextTransfer.getInstance();
      final Object tmpData = aClipboard.getContents(tmpTransfer);
      if (tmpData != null && tmpData instanceof String
          && (tmpEditor instanceof TextCellEditor || tmpEditor instanceof ComboBoxCellEditor)) {
        if (isCellEditorActive()) {
          tmpEditor.performPaste();
        } else {
          final T tmpElement = (T) tmpFocusCell.getViewerRow().getElement();
          setValue(tmpElement, tmpColumnIndex, (String) tmpData);
          informAboutChange(tmpElement);
          refresh();
          setDirty(true);
        }
      }
    }
  }

  private void setValue(final T anElement, final int aColumnIndex, final String aValue) {
    ((AbstractWTETableViewerElement) anElement).set(aColumnIndex, aValue);
  }

  /**
   * Adds a line before the currently selected line.
   */
  public void addLine() {
    final IStructuredSelection tmpSelection = (IStructuredSelection) getSelection();
    if (null != tmpSelection) {
      final T tmpSelectedElement = (T) tmpSelection.getFirstElement();
      if (null != tmpSelectedElement) {
        addNewBefore(tmpSelectedElement);
        refresh();
        // TODO this makes the view<->model discrepancy too visible getTable().setSelection(tmpIndex);
      }
    }
  }

  private void addNewBefore(final T anElement) {
    getContent().addNewElementBefore(anElement);
  }

  /**
   * Deletes the currently selected line and handles row selection.
   */
  public void deleteLine() {
    final IStructuredSelection tmpSelection = (IStructuredSelection) getSelection();
    if (null != tmpSelection) {
      final T tmpSelectedElement = (T) tmpSelection.getFirstElement();
      final int tmpIndex = getElementList().indexOf(tmpSelectedElement);
      if (tmpIndex < getElementList().size() - 1) {
        if (tmpIndex != -1) {
          getElementList().remove(tmpSelectedElement);
          setDirty(true);

          refresh();

          int tmpNewIndex = tmpIndex;
          if (tmpIndex == getElementList().size() - 1) {
            // if this was the last element with content...
            if (getElementList().size() == 1) {
              // ... and the list is empty now
              tmpNewIndex = 0;
            } else {
              tmpNewIndex = tmpIndex - 1;
            }
          }
          getTable().setSelection(tmpNewIndex);

          // TODO cell loses focus -> focus cell
        }
      } else {
        // do not allow to delete the last line
        return;
      }
    }
  }

  /**
   * Moves the currently selected line (and the selection itself) one line down.
   */
  public void moveLineDown() {
    final ViewerCell tmpFocusCell = focusCellManager.getFocusCell();
    if (null != tmpFocusCell) {
      final T tmpElement = (T) tmpFocusCell.getElement();
      final List<T> tmpElementList = getElementList();
      final int tmpMoveIndex = tmpElementList.indexOf(tmpElement);
      if (tmpMoveIndex < getElementList().size() - 1) {
        final T tmpUpper = tmpElementList.get(tmpMoveIndex + 1);
        tmpElementList.set(tmpMoveIndex, tmpUpper);
        tmpElementList.set(tmpMoveIndex + 1, tmpElement);
        setDirty(true);
        getTable().setSelection(tmpMoveIndex);
        refresh();
      }
    }
  }

  /**
   * Moves the currently selected line (and the selection itself) one line up.
   */
  public void moveLineUp() {
    final ViewerCell tmpFocusCell = focusCellManager.getFocusCell();
    if (null != tmpFocusCell) {
      final T tmpElement = (T) tmpFocusCell.getElement();
      final List<T> tmpElementList = getElementList();
      final int tmpMoveIndex = tmpElementList.indexOf(tmpElement);
      if (tmpMoveIndex > 0) {
        final T tmpUpper = tmpElementList.get(tmpMoveIndex - 1);
        tmpElementList.set(tmpMoveIndex, tmpUpper);
        tmpElementList.set(tmpMoveIndex - 1, tmpElement);
        setDirty(true);
        getTable().setSelection(tmpMoveIndex);
        refresh();
      }
    }
  }

  /**
   * @param anElement the element that was changed
   */
  protected void informAboutChange(final T anElement) {
    getContent().elementChanged(anElement);
    setDirty(true);
  }

  /**
   * Provides the content for the {@link org.wetator.testeditor.editors.wte.WTETableViewer}.
   */
  public class WTETableViewerContentProvider implements IStructuredContentProvider {
    @Override
    public void inputChanged(final Viewer aViewer, final Object anOldInput, final Object aNewInput) {
      if (aNewInput != null) {
        ((AbstractWTEContentList<T>) aNewInput).addChangeListener(this);
      }
      if (anOldInput != null && !anOldInput.equals(aNewInput)) {
        ((AbstractWTEContentList<T>) anOldInput).removeChangeListener(this);
      }
    }

    /**
     * @see org.eclipse.jface.viewers.IContentProvider#dispose()
     */
    @Override
    public void dispose() {
      getContent().removeChangeListener(this);
    }

    /**
     * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
     * @param aParent the parent
     * @return all elements
     */
    @Override
    public Object[] getElements(final Object aParent) {
      return getElementList().toArray();
    }

    /**
     * Update the view to reflect the fact that an elemnet was added to the content list.
     *
     * @param anElement the element to add
     */
    public void addElement(final T anElement) {
      add(anElement);
      refresh();
    }

    /**
     * Update the view to reflect the fact that one of the elements was modified.
     *
     * @param anElement the element to update
     */
    public void updateElement(final T anElement) {
      update(anElement, null);
    }
  }

  /**
   * Set the parent editor needed for dirty handling.
   *
   * @param aWetatorTestEditor the WTE this view is running in
   */
  public void setParentEditor(final WetatorTestMultiPageEditor aWetatorTestEditor) {
    wte = aWetatorTestEditor;
  }

  /**
   * @param aDirty the dirty to set
   */
  public void setDirty(final boolean aDirty) {
    dirty = aDirty;
    wte.changedDirty();
  }

  /**
   * @return true if the content was changed since last save, false if not.
   */
  public boolean isDirty() {
    return dirty;
  }

  /**
   * Repaints the table viewer.
   */
  public void repaint() {
    resizeDynamicWidthColumns();
  }

  /**
   * Resizes the columns with dynamic width.
   */
  protected abstract void resizeDynamicWidthColumns();

  /**
   * @return the parent composite of the table
   */
  public Composite getTableParent() {
    return getTable().getParent();
  }

  /**
   * Reloads the input.
   */
  public void reloadInput() {
    setInput(getContent());
  }

  /**
   * @see org.wetator.testeditor.editors.wte.IWTETableViewer#getContentManager()
   * @return the contentManager
   */
  public WetatorTestContentManager getContentManager() {
    return contentManager;
  }

  /**
   * Return the column names in a collection.
   *
   * @return list containing column names
   */
  public abstract List<String> getColumnNames();

  /**
   * @param aColumnIndex the index of the column
   * @return the cell label provider
   */
  protected abstract ColumnLabelProvider getCellLabelProvider(int aColumnIndex);

  /**
   * @return the content from the model handled by the table viewer
   */
  protected abstract AbstractWTEContentList<T> getContent();

  /**
   * @return the {@link List} object from the {@link AbstractWTEContentList}
   */
  protected List<T> getElementList() {
    return getContent().getElements();
  }
}
