/*
 * Copyright (c) 2008-2016 wetator.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package org.wetator.testeditor.editors.xsd;

import java.util.Arrays;
import java.util.List;

import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.wetator.testeditor.editors.AbstractWTEContentList;
import org.wetator.testeditor.editors.AbstractWTETableViewer;
import org.wetator.testeditor.editors.WTEMessages;
import org.wetator.testeditor.editors.WetatorTestContentManager;

/**
 * The table viewer for XSD Wetator schema files.
 * 
 * @author tobwoerk
 * @author frank.danek
 */
public class WTEXSDTableViewer extends AbstractWTETableViewer<WetatorXMLSchema> {

  /**
   * Index of the prefix column.
   */
  public static final int PREFIX = 0;
  /**
   * Index of the namespace column.
   */
  public static final int NAMESPACE = 1;
  /**
   * Index of the location column.
   */
  public static final int LOCATION = 2;

  private static String[] columnNames = new String[] { WTEMessages.headerColumnPrefix,
      WTEMessages.headerColumnNamespace, WTEMessages.headerColumnLocation };

  /**
   * Constructor.
   * 
   * @param aParent the parent
   * @param aContentManager the contentManager holding the test content
   */
  public WTEXSDTableViewer(final Composite aParent, final WetatorTestContentManager aContentManager) {
    super(aParent, aContentManager);
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.wetator.testeditor.editors.AbstractWTETableViewer#createColumns()
   */
  @Override
  protected void createColumns() {
    TableViewerColumn tmpColumn = createColumn(PREFIX, SWT.CENTER);
    tmpColumn.setEditingSupport(new AbstractWTEXSDEditingSupport(this, PREFIX) {
      @Override
      protected void setValue(final WetatorXMLSchema aSchema, final Object aValue) {
        final String tmpValue = (String) aValue;
        if (!tmpValue.equals(aSchema.getPrefix())) {
          if (tmpValue.equals(WetatorTestContentManager.SQL_COMMAND_SET_SCHEMA.getPrefix())) {
            aSchema.setNamespace(WetatorTestContentManager.SQL_COMMAND_SET_SCHEMA.getNamespace());
            aSchema.setLocation(WetatorTestContentManager.SQL_COMMAND_SET_SCHEMA.getLocation());
          } else if (tmpValue.equals(WetatorTestContentManager.TEST_COMMAND_SET_SCHEMA.getPrefix())) {
            aSchema.setNamespace(WetatorTestContentManager.TEST_COMMAND_SET_SCHEMA.getNamespace());
            aSchema.setLocation(WetatorTestContentManager.TEST_COMMAND_SET_SCHEMA.getLocation());
          } else if (tmpValue.equals(WetatorTestContentManager.INCUBATOR_COMMAND_SET_SCHEMA.getPrefix())) {
            aSchema.setNamespace(WetatorTestContentManager.INCUBATOR_COMMAND_SET_SCHEMA.getNamespace());
            aSchema.setLocation(WetatorTestContentManager.INCUBATOR_COMMAND_SET_SCHEMA.getLocation());
          }
          aSchema.setPrefix(tmpValue);
          informAboutChange(aSchema);
        }
      }

      @Override
      protected Object getValue(final WetatorXMLSchema aSchema) {
        return aSchema.getPrefix();
      }
    });
    tmpColumn = createColumn(NAMESPACE, SWT.LEFT);
    tmpColumn.setEditingSupport(new AbstractWTEXSDEditingSupport(this, NAMESPACE) {
      @Override
      protected void setValue(final WetatorXMLSchema aSchema, final Object aValue) {
        final String tmpValue = (String) aValue;
        if (!tmpValue.equals(aSchema.getNamespace())) {
          aSchema.setNamespace(tmpValue);
          informAboutChange(aSchema);
        }
      }

      @Override
      protected Object getValue(final WetatorXMLSchema aSchema) {
        return aSchema.getNamespace();
      }
    });
    tmpColumn = createColumn(LOCATION, SWT.LEFT);
    tmpColumn.setEditingSupport(new AbstractWTEXSDEditingSupport(this, LOCATION) {
      @Override
      protected void setValue(final WetatorXMLSchema aSchema, final Object aValue) {
        final String tmpValue = (String) aValue;
        if (!tmpValue.equals(aSchema.getLocation())) {
          aSchema.setLocation(tmpValue);
          informAboutChange(aSchema);
        }
      }

      @Override
      protected Object getValue(final WetatorXMLSchema aSchema) {
        return aSchema.getLocation();
      }
    });
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.wetator.testeditor.editors.AbstractWTETableViewer#initializeEditors(org.eclipse.jface.viewers.CellEditor[])
   */
  @Override
  protected void initializeEditors(final CellEditor[] anEditorArray) {
    CellEditor tmpEditor = new TextCellEditor(getTable());
    anEditorArray[PREFIX] = tmpEditor;

    tmpEditor = new TextCellEditor(getTable());
    anEditorArray[NAMESPACE] = tmpEditor;

    tmpEditor = new TextCellEditor(getTable());
    anEditorArray[LOCATION] = tmpEditor;
  }

  /**
   * Checks if the two mandatory XSDs are effected. If not {@inheritDoc}
   * 
   * @see org.wetator.testeditor.editors.AbstractWTETableViewer#addLine()
   */
  @Override
  public void addLine() {
    if (!mandatoryXSDsEffected(1)) {
      super.addLine();
    }
  }

  /**
   * Checks if the two mandatory XSDs are effected. If not {@inheritDoc}
   * 
   * @see org.wetator.testeditor.editors.AbstractWTETableViewer#moveLineDown()
   */
  @Override
  public void moveLineDown() {
    if (!mandatoryXSDsEffected(1)) {
      super.moveLineDown();
    }
  }

  /**
   * Checks if the two mandatory XSDs are effected. If not {@inheritDoc}
   * 
   * @see org.wetator.testeditor.editors.AbstractWTETableViewer#moveLineUp()
   */
  @Override
  public void moveLineUp() {
    if (!mandatoryXSDsEffected(2)) {
      super.moveLineUp();
    }
  }

  private boolean mandatoryXSDsEffected(final int aCheckIndex) {
    final IStructuredSelection tmpSelection = (IStructuredSelection) getSelection();
    if (null != tmpSelection) {
      final WetatorXMLSchema tmpWetatorXMLSchema = (WetatorXMLSchema) tmpSelection.getFirstElement();
      final int tmpIndex = getElementList().indexOf(tmpWetatorXMLSchema);
      if (tmpIndex > aCheckIndex) {
        return false;
      }
      return true;
    }
    return false;
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.wetator.testeditor.editors.AbstractWTETableViewer#repaint()
   */
  @Override
  public void repaint() {
    // adjust prefix column
    getTable().getColumn(PREFIX).setWidth(60);
    resizeDynamicWidthColumns();
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.wetator.testeditor.editors.AbstractWTETableViewer#resizeDynamicWidthColumns()
   */
  @Override
  protected void resizeDynamicWidthColumns() {
    final Table tmpTable = getTable();
    // determine parameter width
    final TableColumn[] tmpColumns = tmpTable.getColumns();
    int tmpSize = 0;
    for (final TableColumn tmpColumn : tmpColumns) {
      final int tmpIndex = tmpTable.indexOf(tmpColumn);
      if (tmpIndex == PREFIX) {
        tmpSize += tmpColumn.getWidth();
        tmpSize += 8;
      }
    }

    final int tmpColSize = (tmpTable.getSize().x - tmpSize) / 2;
    tmpTable.getColumn(NAMESPACE).setWidth(tmpColSize);
    tmpTable.getColumn(LOCATION).setWidth(tmpColSize);
  }

  /**
   * Return the column names in a collection.
   * 
   * @return List containing column names
   */
  @Override
  public List<String> getColumnNames() {
    return Arrays.asList(columnNames);
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.wetator.testeditor.editors.AbstractWTETableViewer#getCellLabelProvider(int)
   */
  @Override
  protected ColumnLabelProvider getCellLabelProvider(final int aColumnIndex) {
    return new WTEXSDLabelProvider(aColumnIndex);
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.wetator.testeditor.editors.AbstractWTETableViewer#getContent()
   */
  @Override
  protected AbstractWTEContentList<WetatorXMLSchema> getContent() {
    return contentManager.getSchemaList();
  }
}