/*
 * Copyright (c) 2008-2016 wetator.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package org.wetator.testeditor.editors.wte;

import java.util.Arrays;
import java.util.List;

import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.CheckboxCellEditor;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.ColumnViewerEditorActivationEvent;
import org.eclipse.jface.viewers.ColumnViewerToolTipSupport;
import org.eclipse.jface.viewers.ComboBoxCellEditor;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.viewers.ViewerCell;
import org.eclipse.jface.window.ToolTip;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.wetator.testeditor.editors.AbstractWTEContentList;
import org.wetator.testeditor.editors.AbstractWTETableViewer;
import org.wetator.testeditor.editors.WTEMessages;
import org.wetator.testeditor.editors.WetatorTestContentManager;

/**
 * The table viewer for Wetator tests.
 * 
 * @author tobwoerk
 * @author frank.danek
 */
public class WTETableViewer extends AbstractWTETableViewer<WetatorCommand> implements IWTETableViewer {

  /**
   * Index of the "isComment" column.
   */
  public static final int IS_COMMENT = 0;
  /**
   * Index of the command column.
   */
  public static final int COMMAND = 1;
  /**
   * Index of the parameter column.
   */
  public static final int PARAMETER = 2;
  /**
   * Index of the optional parameter column.
   */
  public static final int OPTIONAL_PARAMETER = 3;
  /**
   * Index of the optional parameter 2 column.
   */
  public static final int OPTIONAL_PARAMETER2 = 4;

  private static String[] columnNames = new String[] { "", WTEMessages.headerColumnCommand,
      WTEMessages.headerColumnParameter, WTEMessages.headerColumnOptionalParameter,
      WTEMessages.headerColumnOptionalParameter2 };

  private boolean showOptionalParameter2;

  /**
   * Constructor.
   * 
   * @param aParent the parent
   * @param aContentManager the contentManager holding the test content
   */
  public WTETableViewer(final Composite aParent, final WetatorTestContentManager aContentManager) {
    super(aParent, aContentManager);
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.wetator.testeditor.editors.AbstractWTETableViewer#createColumns()
   */
  @Override
  protected void createColumns() {
    TableViewerColumn tmpColumn = createColumn(IS_COMMENT, SWT.CENTER);
    tmpColumn.setEditingSupport(new AbstractWTEEditingSupport(this, IS_COMMENT) {
      @Override
      protected void setValue(final WetatorCommand aCommand, final Object aValue) {
        final Boolean tmpValue = (Boolean) aValue;
        if (!aValue.equals(aCommand.isComment())) {
          aCommand.setComment(tmpValue.booleanValue());
          informAboutChange(aCommand);
        }
      }

      @Override
      protected Object getValue(final WetatorCommand aCommand) {
        return aCommand.isComment();
      }
    });
    tmpColumn = createColumn(COMMAND, SWT.CENTER);
    tmpColumn.setEditingSupport(new AbstractWTEEditingSupport(this, COMMAND) {
      @Override
      protected void setValue(final WetatorCommand aCommand, final Object aValue) {
        final String tmpValue = contentManager.getAvailableCommands()[((Integer) aValue).intValue()];
        if (!tmpValue.equals(aCommand.getCommandName())) {
          aCommand.setCommand(tmpValue);
          informAboutChange(aCommand);
        }
      }

      @Override
      protected Object getValue(final WetatorCommand aCommand) {
        final String tmpCommandString = aCommand.getCommandName();
        final String[] tmpChoices = contentManager.getAvailableCommands();
        int i = tmpChoices.length - 1;
        while (!tmpCommandString.equals(tmpChoices[i]) && i > 0) {
          i--;
        }
        return Integer.valueOf(i);
      }
    });
    tmpColumn = createColumn(PARAMETER, SWT.LEFT);
    tmpColumn.setEditingSupport(new AbstractWTEEditingSupport(this, PARAMETER) {
      @Override
      protected void setValue(final WetatorCommand aCommand, final Object aValue) {
        final String tmpValue = (String) aValue;
        if (!tmpValue.equals(aCommand.getParameter())) {
          aCommand.setParameter(tmpValue);
          informAboutChange(aCommand);
        }
      }

      @Override
      protected Object getValue(final WetatorCommand aCommand) {
        return aCommand.getParameter();
      }
    });
    tmpColumn = createColumn(OPTIONAL_PARAMETER, SWT.LEFT);
    tmpColumn.setEditingSupport(new AbstractWTEEditingSupport(this, OPTIONAL_PARAMETER) {
      @Override
      protected void setValue(final WetatorCommand aCommand, final Object aValue) {
        final String tmpValue = (String) aValue;
        if (!tmpValue.equals(aCommand.getOptionalParameter())) {
          aCommand.setOptionalParameter(tmpValue);
          informAboutChange(aCommand);
        }
      }

      @Override
      protected Object getValue(final WetatorCommand aCommand) {
        return aCommand.getOptionalParameter();
      }
    });
    tmpColumn = createColumn(OPTIONAL_PARAMETER2, SWT.LEFT);
    tmpColumn.setEditingSupport(new AbstractWTEEditingSupport(this, OPTIONAL_PARAMETER2) {
      @Override
      protected void setValue(final WetatorCommand aCommand, final Object aValue) {
        final String tmpValue = (String) aValue;
        if (!tmpValue.equals(aCommand.getOptionalParameter2())) {
          aCommand.setOptionalParameter2(tmpValue);
          informAboutChange(aCommand);
        }
      }

      @Override
      protected Object getValue(final WetatorCommand aCommand) {
        return aCommand.getOptionalParameter2();
      }

      @Override
      protected boolean canEdit(final WetatorCommand aCommand) {
        if (!aCommand.hasData() //
            || !"".equals(aCommand.getOptionalParameter2())
            && contentManager.isAnyCommandTypeWith3Parameters()
            || contentManager.isCommandTypeWith3Parameters(aCommand.getCommandName())) {
          return true;
        }
        return false;
      }
    });
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.wetator.testeditor.editors.AbstractWTETableViewer#initializeEditors(org.eclipse.jface.viewers.CellEditor[])
   */
  @Override
  protected void initializeEditors(final CellEditor[] anEditorArray) {
    CellEditor tmpEditor = new CheckboxCellEditor(getTable());
    anEditorArray[IS_COMMENT] = tmpEditor;

    final ComboBoxCellEditor tmpCommandEditor = new ComboBoxCellEditor(getTable(),
        contentManager.getAvailableCommands(), SWT.READ_ONLY | SWT.DROP_DOWN);
    tmpCommandEditor.setActivationStyle(ComboBoxCellEditor.DROP_DOWN_ON_KEY_ACTIVATION
        | ComboBoxCellEditor.DROP_DOWN_ON_MOUSE_ACTIVATION | ComboBoxCellEditor.DROP_DOWN_ON_PROGRAMMATIC_ACTIVATION);
    anEditorArray[COMMAND] = tmpCommandEditor;

    tmpEditor = new TextCellEditor(getTable());
    anEditorArray[PARAMETER] = tmpEditor;

    tmpEditor = new TextCellEditor(getTable());
    anEditorArray[OPTIONAL_PARAMETER] = tmpEditor;

    tmpEditor = new TextCellEditor(getTable());
    anEditorArray[OPTIONAL_PARAMETER2] = tmpEditor;
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.wetator.testeditor.editors.AbstractWTETableViewer#initialize()
   */
  @Override
  protected void initialize() {
    ColumnViewerToolTipSupport.enableFor(this, ToolTip.NO_RECREATE);
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.wetator.testeditor.editors.AbstractWTETableViewer#isActivatingEditor(org.eclipse.jface.viewers.ColumnViewerEditorActivationEvent)
   */
  @Override
  protected boolean isActivatingEditor(final ColumnViewerEditorActivationEvent anEvent) {
    final ViewerCell tmpCell = (ViewerCell) anEvent.getSource();
    final int tmpIndex = tmpCell.getColumnIndex();

    if (IS_COMMENT == tmpIndex) {
      return anEvent.eventType == ColumnViewerEditorActivationEvent.MOUSE_CLICK_SELECTION
          || anEvent.eventType == ColumnViewerEditorActivationEvent.MOUSE_DOUBLE_CLICK_SELECTION
          || anEvent.eventType == ColumnViewerEditorActivationEvent.KEY_PRESSED
          && (anEvent.keyCode == SWT.CR || anEvent.keyCode == SWT.F2 || anEvent.character == ' ')
          || anEvent.eventType == ColumnViewerEditorActivationEvent.PROGRAMMATIC;
    }
    return super.isActivatingEditor(anEvent);
  }

  /**
   * Repaints the page.
   */
  @Override
  public void repaint() {
    // reload available commands for drop down
    final ComboBoxCellEditor tmpCommandEditor = (ComboBoxCellEditor) getCellEditors()[COMMAND];
    if (null != tmpCommandEditor) {
      tmpCommandEditor.setItems(contentManager.getAvailableCommands());
    }

    // adjust comment column
    getTable().getColumn(IS_COMMENT).setWidth(20);

    // pack command column
    final TableColumn tmpCommandColumn = getTable().getColumn(COMMAND);
    tmpCommandColumn.pack();
    tmpCommandColumn.setWidth(tmpCommandColumn.getWidth() + 50);

    resizeDynamicWidthColumns();
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.wetator.testeditor.editors.AbstractWTETableViewer#resizeDynamicWidthColumns()
   */
  @Override
  protected void resizeDynamicWidthColumns() {
    final Table tmpTable = getTable();
    // determine parameter width
    final TableColumn[] tmpColumns = tmpTable.getColumns();
    int tmpSize = 0;
    for (final TableColumn tmpColumn : tmpColumns) {
      final int tmpIndex = tmpTable.indexOf(tmpColumn);
      if (tmpIndex != PARAMETER && tmpIndex != OPTIONAL_PARAMETER && tmpIndex != OPTIONAL_PARAMETER2) {
        tmpSize += tmpColumn.getWidth();
        tmpSize += 8;
      }
    }

    if (showOptionalParameter2) {
      final int tmpColSize = (tmpTable.getSize().x - tmpSize) / 4;
      tmpTable.getColumn(PARAMETER).setWidth(tmpColSize + tmpColSize / 2);
      tmpTable.getColumn(OPTIONAL_PARAMETER).setWidth(tmpColSize + tmpColSize / 2);
      tmpTable.getColumn(OPTIONAL_PARAMETER2).setWidth(tmpColSize);
    } else {
      final int tmpColSize = (tmpTable.getSize().x - tmpSize) / 2;
      tmpTable.getColumn(PARAMETER).setWidth(tmpColSize);
      tmpTable.getColumn(OPTIONAL_PARAMETER).setWidth(tmpColSize);
      tmpTable.getColumn(OPTIONAL_PARAMETER2).setWidth(0);
    }
  }

  /**
   * Return the column names in a collection.
   * 
   * @return List containing column names
   */
  @Override
  public List<String> getColumnNames() {
    return Arrays.asList(columnNames);
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.wetator.testeditor.editors.AbstractWTETableViewer#getCellLabelProvider(int)
   */
  @Override
  protected ColumnLabelProvider getCellLabelProvider(final int aColumnIndex) {
    return new WTELabelProvider(aColumnIndex, contentManager);
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.wetator.testeditor.editors.AbstractWTETableViewer#getContent()
   */
  @Override
  protected AbstractWTEContentList<WetatorCommand> getContent() {
    return contentManager.getCommandList();
  }

  /**
   * @param aShowOptionalParameter2 the showOptionalParameter2 to set
   */
  public void setShowOptionalParameter2(final boolean aShowOptionalParameter2) {
    showOptionalParameter2 = aShowOptionalParameter2;
  }
}