/*
 * Copyright (c) 2008-2016 wetator.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package org.wetator.testeditor.editors;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import javax.xml.stream.XMLStreamException;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.editors.text.TextEditor;
import org.eclipse.ui.ide.FileStoreEditorInput;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.part.FileEditorInput;
import org.eclipse.ui.part.MultiPageEditorPart;
import org.wetator.exception.InvalidInputException;
import org.wetator.testeditor.editors.exception.CommandNotSupportedException;
import org.wetator.testeditor.editors.exception.EclipseErrorUtil;
import org.wetator.testeditor.editors.exception.FileNotSupportedException;
import org.wetator.testeditor.editors.wte.WTETableViewer;
import org.wetator.testeditor.editors.xml.XMLEditor;
import org.wetator.testeditor.editors.xsd.WTEXSDTableViewer;

/**
 * The WTE {@link MultiPageEditorPart}.
 * <ul>
 * <li>page 0 contains the real WTE ({@link WTETableViewer}).</li>
 * <li>page 1 contains the XSD schema selection editor ({@link WTEXSDTableViewer}).</li>
 * <li>page 2 contains a simple XML editor ({@link XMLEditor}).</li>
 * </ul>
 * 
 * @author tobwoerk
 * @author frank.danek
 */
public class WetatorTestMultiPageEditor extends MultiPageEditorPart implements IResourceChangeListener {

  /** The {@link WTETableViewer}. */
  public WTETableViewer wte;
  private static final String TITLE_WTE = "WTE";
  /** The index of the {@link WTETableViewer}. */
  public int indexWTE;

  /** The {@link WTEXSDTableViewer}. */
  public WTEXSDTableViewer xsd;
  private static final String TITLE_XSD = "XSD";
  /** The index of the {@link WTEXSDTableViewer}. */
  public int indexXSDEditor;

  private TextEditor xmlEditor;
  private static final String TITLE_XML_EDITOR = "Source";
  private int indexXMLEditor;

  private int currentPage;
  private boolean wteProblem;

  private boolean autoConverted;

  private final WetatorTestContentManager contentManager;

  /**
   * Creates the {@link WetatorTestMultiPageEditor}.
   */
  public WetatorTestMultiPageEditor() {
    super();
    contentManager = new WetatorTestContentManager();
    ResourcesPlugin.getWorkspace().addResourceChangeListener(this);
  }

  /**
   * Creates the pages of the {@link WetatorTestMultiPageEditor}.
   */
  @Override
  protected void createPages() {
    try {
      initializeContentManager();
      createWetatorTestEditor();
      wte.setDirty(autoConverted);
      createWetatorXSDEditor();
      createXmlEditorPage();
    } catch (final InvalidInputException e) {
      showErrorPopUp(e, "File not supported",
          "File cannot be opened.\n\nError while scripting file. Please check the validity of your test file.");
      closeEditor();
    } catch (final FileNotSupportedException e) {
      showErrorPopUp(e, "File not supported",
          "File cannot be opened.\n\nThe WetatorTestEditor only supports the Wetator build-in XML test file formats.");
      closeEditor();
    } catch (final CommandNotSupportedException e) {
      showErrorPopUp(
          e,
          "Auto conversion failed",
          "File (old XML test-case format) cannot be converted.\n\nAt the moment the WetatorTestEditor only supports auto conversion for test cases which only make use of Wetator build-in command sets.");
      closeEditor();
    } catch (final Exception e) {
      EclipseErrorUtil.logError(e, this.getClass().getSimpleName());
      closeEditor();
    }
  }

  private void initializeContentManager() throws InvalidInputException, FileNotSupportedException,
      CommandNotSupportedException, XMLStreamException, IOException {
    if (!contentManager.initialize(getFilePath())) {
      // initialize again (in correct format now)
      contentManager.initialize(true, getFilePath(), contentManager.getTestCaseAsXml());
      autoConverted = true;
    }
  }

  private String getFilePath() {
    String tmpPath;
    final IEditorInput tmpInput = getEditorInput();
    if (tmpInput instanceof IFileEditorInput) {
      tmpPath = ((IFileEditorInput) tmpInput).getFile().getLocationURI().getPath();
    } else if (tmpInput instanceof FileStoreEditorInput) {
      tmpPath = ((FileStoreEditorInput) tmpInput).getURI().getPath();
    } else {
      throw new RuntimeException("Invalid input: Must be IFileEditorInput or FileStoreEditorInput");
    }
    return tmpPath;
  }

  /**
   * Creates the WTE ({@link WTETableViewer}).
   */
  private void createWetatorTestEditor() {
    final Composite tmpComposite = new Composite(getContainer(), SWT.ALL);
    wte = new WTETableViewer(tmpComposite, contentManager);
    wte.setParentEditor(this);
    indexWTE = addPage(wte.getTableParent());
    setPageText(indexWTE, TITLE_WTE);
  }

  /**
   * Creates the WTE ({@link WTEXSDTableViewer}).
   */
  private void createWetatorXSDEditor() {
    final Composite tmpComposite = new Composite(getContainer(), SWT.ALL);
    xsd = new WTEXSDTableViewer(tmpComposite, contentManager);
    xsd.setParentEditor(this);
    indexXSDEditor = addPage(xsd.getTableParent());
    setPageText(indexXSDEditor, TITLE_XSD);
  }

  /**
   * Creates the XML editor page.
   */
  private void createXmlEditorPage() {
    try {
      xmlEditor = new XMLEditor();
      indexXMLEditor = addPage(xmlEditor, getEditorInput());
      setPageText(indexXMLEditor, TITLE_XML_EDITOR);
    } catch (final PartInitException e) {
      EclipseErrorUtil.logError(e, this.getClass().getSimpleName());
      showErrorPopUp(e, "Error creating the XML editor", "The XML editor page could not be created");
    }
  }

  /**
   * @see org.eclipse.ui.part.MultiPageEditorPart#dispose()
   */
  @Override
  public void dispose() {
    ResourcesPlugin.getWorkspace().removeResourceChangeListener(this);
    super.dispose();
  }

  private void closeEditor() {
    getSite().getShell().getDisplay().asyncExec(new Runnable() {
      @Override
      public void run() {
        getSite().getPage().closeEditor(WetatorTestMultiPageEditor.this, false);
      }
    });
  }

  private void showErrorPopUp(final Exception anException, final String aTitle, final String aMessage) {
    final Status tmpStatus = EclipseErrorUtil.createErrorStatus(anException, this.getClass().getSimpleName());
    ErrorDialog.openError(getSite().getShell(), aTitle, aMessage, tmpStatus);
  }

  private boolean checkBeforeSave() {
    try {
      if (currentPage != indexXMLEditor) {
        reloadDataForXMLEditor();
      }
      final String tmpContent = xmlEditor.getDocumentProvider().getDocument(xmlEditor.getEditorInput()).get();
      contentManager.initialize(isDirty(), getFilePath(), tmpContent);
    } catch (final Exception e) {
      showErrorPopUp(e, "Could not save file",
          "The file content is invalid.\nPlease correct it before saving the test case.");
      return false;
    }
    return true;
  }

  /**
   * Saves the multi-page editor's document.
   * 
   * @param aMonitor monitor for the progress
   */
  @Override
  public void doSave(final IProgressMonitor aMonitor) {
    if (!checkBeforeSave()) {
      return;
    }

    try {
      save(aMonitor);
    } catch (final Exception e) {
      showErrorPopUp(e, "Could not save file", "Refreshing the XML content before saving encountered a problem.");
    }
    refreshActiveTableViewer();
  }

  private void save(final IProgressMonitor aMonitor) {
    xmlEditor.doSave(aMonitor);
    wte.setDirty(super.isDirty());
    xsd.setDirty(super.isDirty());
  }

  private void refreshActiveTableViewer() {
    if (indexWTE == getActivePage()) {
      wte.refresh();
    } else if (indexXSDEditor == getActivePage()) {
      xsd.refresh();
    }
  }

  /**
   * @see org.eclipse.ui.part.EditorPart#doSaveAs()
   */
  @Override
  public void doSaveAs() {
    if (!checkBeforeSave()) {
      return;
    }

    try {
      final IEditorPart tmpEditor = getEditor(indexXMLEditor);
      tmpEditor.doSaveAs();
      try {
        reloadDataForWTE();
        reloadDataForXSD();
        reloadDataForXMLEditor();
        setPartName(xmlEditor.getEditorInput().getName());
        setPathAsContentDescription();
        wte.setDirty(super.isDirty());
        xsd.setDirty(super.isDirty());
      } catch (final Exception e) {
        // unexpected
        EclipseErrorUtil.logError(e, this.getClass().getSimpleName());
        showErrorPopUp(e, e.getClass().getSimpleName(), "Unexpected exception while saving as.");
      }
    } catch (final Exception e) {
      showErrorPopUp(e, "Could not save file", "Refreshing the XML content before saving as encountered a problem.");
    }
    refreshActiveTableViewer();
  }

  /**
   * @param aMarker the marker to go to
   */
  public void gotoMarker(final IMarker aMarker) {
    setActivePage(0);
    IDE.gotoMarker(getEditor(0), aMarker);
  }

  /**
   * Initializes the WTE.
   * 
   * @param aSite the site to initialize
   * @param anEditorInput the editor input
   * @throws PartInitException in case of invalid editor input
   */
  @Override
  public void init(final IEditorSite aSite, final IEditorInput anEditorInput) throws PartInitException {
    if (anEditorInput instanceof IFileEditorInput || anEditorInput instanceof FileStoreEditorInput) {
      super.init(aSite, anEditorInput);
      setPartName(anEditorInput.getName());
      setPathAsContentDescription();
      return;
    }
    throw new PartInitException("Invalid Input: Must be IFileEditorInput or FileStoreEditorInput");
  }

  private void setPathAsContentDescription() {
    String tmpPath;
    if (getEditorInput() instanceof IFileEditorInput) {
      tmpPath = ((IFileEditorInput) getEditorInput()).getFile().getFullPath().toOSString();
    } else {
      tmpPath = ((FileStoreEditorInput) getEditorInput()).getURI().getPath();
    }
    setContentDescription(tmpPath);
  }

  /**
   * @see org.eclipse.ui.part.EditorPart#isSaveAsAllowed()
   * @return always true
   */
  @Override
  public boolean isSaveAsAllowed() {
    return true;
  }

  /**
   * @see org.eclipse.ui.part.MultiPageEditorPart#pageChange(int)
   * @param aNewPageIndex the index of the page to change to
   */
  @Override
  protected void pageChange(final int aNewPageIndex) {
    try {
      if (!wteProblem) {
        if (currentPage != indexXMLEditor) {
          reloadDataForXMLEditor();
        }
        final String tmpContent = xmlEditor.getDocumentProvider().getDocument(xmlEditor.getEditorInput()).get();
        contentManager.initialize(isDirty(), getFilePath(), tmpContent);
      }

      if (aNewPageIndex == indexWTE) {
        wte.setDirty(super.isDirty() || xsd.isDirty());
        try {
          reloadDataForWTE();
        } catch (final CommandNotSupportedException e) {
          wteProblem = true; // we can (try to) handle this
          showErrorPopUp(
              e,
              "Command not supported",
              "Validation of commands failed.\n\nPlease adjust the wrong commands or the missing XSD information in the XML editor page.");
        } catch (final Exception e) {
          wteProblem = true; // we can (try to) handle this
          showErrorPopUp(e, "Error refreshing the WTE",
              "The WTE page could not be created.\n\nPlease correct the content in the XML editor page.");
        }
        if (wteProblem) {
          setActiveEditor(xmlEditor);
        } else {
          currentPage = indexWTE;
          wte.setDirty(super.isDirty() || xsd.isDirty());
          wte.repaint();
        }
      } else if (aNewPageIndex == indexXSDEditor) {
        currentPage = indexXSDEditor;
        xsd.setDirty(super.isDirty() || wte.isDirty());
        reloadDataForXSD();
        xsd.repaint();
      } else if (aNewPageIndex == indexXMLEditor) {
        currentPage = indexXMLEditor;
        if (wteProblem) {
          // reset and let the user work on the problems
          wteProblem = false;
        } else {
          reloadDataForXMLEditor();
        }
      }
    } catch (final InvalidInputException e) {
      wteProblem = true; // we can (try to) handle this
      showErrorPopUp(e, "Error refreshing the WTE",
          "The file content is invalid.\nPlease correct it in the XML editor page.");
      setActiveEditor(xmlEditor);
    } catch (final Exception e) {
      // we cannot handle these
      EclipseErrorUtil.logError(e, this.getClass().getSimpleName());
      closeEditor();
    }
    super.pageChange(aNewPageIndex);
  }

  private void reloadDataForWTE() throws Exception {
    wte.reloadInput();
    wte.setShowOptionalParameter2(contentManager.isAnyCommandTypeWith3Parameters());
  }

  private void reloadDataForXSD() throws Exception {
    xsd.reloadInput();
  }

  private void reloadDataForXMLEditor() throws Exception {
    try {
      final String tmpNewContent = contentManager.getTestCaseAsXml();
      if (resourceChanged(tmpNewContent)) {
        xmlEditor.getDocumentProvider().getDocument(xmlEditor.getEditorInput()).set(tmpNewContent);
      }
    } catch (final Exception e) {
      EclipseErrorUtil.logError(e, this.getClass().getSimpleName());
      showErrorPopUp(e, "Error refreshing the XML editor",
          "The XML editor page could not be created.\n\nThe current content is not a valid Wetator test case.");
      throw e;
    }
  }

  private boolean resourceChanged(final String aNewContent) {
    final String tmpOldContent = xmlEditor.getDocumentProvider().getDocument(xmlEditor.getEditorInput()).get();
    if (aNewContent.equals(tmpOldContent)) {
      return false;
    }
    return true;
  }

  /**
   * Recognizes file name changes and closes all project files on project close.
   * 
   * @param anEvent the change event
   */
  @Override
  public void resourceChanged(final IResourceChangeEvent anEvent) {
    if (anEvent.getType() == IResourceChangeEvent.POST_CHANGE) {
      final IResourceDelta tmpDelta = anEvent.getDelta();

      if (IResourceDelta.CHANGED == tmpDelta.getKind()) {
        String tmpNewPath = null;
        String tmpNewFileName = null;
        IEditorInput tmpInput = null;
        final List<IResourceDelta> tmpChildren = getEndChildren(tmpDelta);
        for (final IResourceDelta tmpChild : tmpChildren) {
          if (tmpChild.getFullPath().toOSString().equals(getContentDescription())
              && (tmpChild.getFlags() & IResourceDelta.MOVED_TO) != 0) {
            tmpNewPath = tmpChild.getMovedToPath().toString();
            tmpNewFileName = tmpNewPath.substring(tmpNewPath.lastIndexOf('/') + 1);
            final File tmpFile = new File(tmpNewPath);
            final IPath tmpLocation = Path.fromOSString(tmpFile.getAbsolutePath());
            final IFile tmpIFile = ResourcesPlugin.getWorkspace().getRoot().getFile(tmpLocation);

            tmpInput = new FileEditorInput(tmpIFile);
            break;
          }
        }

        if (null != tmpNewFileName) {
          final IEditorInput tmpFinalInput = tmpInput;
          Display.getDefault().asyncExec(new Runnable() {
            @Override
            public void run() {
              try {
                init(getEditorSite(), tmpFinalInput);
              } catch (final PartInitException e) {
                EclipseErrorUtil.logError(e, this.getClass().getSimpleName());
                showErrorPopUp(e, e.getClass().getSimpleName(), "Unexpected exception while reinitializing editor.");
              }
            }
          });
        }
      }
    } else if (anEvent.getType() == IResourceChangeEvent.PRE_CLOSE) {
      Display.getDefault().asyncExec(new Runnable() {
        @Override
        public void run() {
          final IWorkbenchPage[] tmpPages = getSite().getWorkbenchWindow().getPages();
          for (int i = 0; i < tmpPages.length; i++) {
            if (((FileEditorInput) xmlEditor.getEditorInput()).getFile().getProject().equals(anEvent.getResource())) {
              final IEditorPart tmpEditorPart = tmpPages[i].findEditor(xmlEditor.getEditorInput());
              tmpPages[i].closeEditor(tmpEditorPart, true);
            }
          }
        }
      });
    }
  }

  private List<IResourceDelta> getEndChildren(final IResourceDelta aDelta) {
    final List<IResourceDelta> tmpEndChildren = new ArrayList<IResourceDelta>();

    final IResourceDelta[] tmpChildren = aDelta.getAffectedChildren();
    for (final IResourceDelta tmpDeltaChild : tmpChildren) {
      if (tmpDeltaChild.getAffectedChildren().length == 0) {
        tmpEndChildren.add(tmpDeltaChild);
      } else {
        tmpEndChildren.addAll(getEndChildren(tmpDeltaChild));
      }
    }

    return tmpEndChildren;
  }

  /**
   * Fires the dirty property change event.
   */
  public void changedDirty() {
    firePropertyChange(PROP_DIRTY);
  }

  /**
   * @see org.eclipse.ui.part.MultiPageEditorPart#isDirty()
   * @return true if dirty, false if not
   */
  @Override
  public boolean isDirty() {
    return super.isDirty() || wte.isDirty() || xsd.isDirty();
  }

  /**
   * @return the contentManager
   */
  public WetatorTestContentManager getContentManager() {
    return contentManager;
  }
}
